#!/usr/bin/env python3
from datetime import datetime
import copy
import gzip
import json
import matplotlib.pyplot as plt
import sys

IGNORE = {
    "none" : [], # Do not ignore anything

    "MinMax": [ "line.10:reactive_power_from", "line.10:current_from", "trafo.1:reactive_power_lv", "sgen.10:target_active_power", "sgen.10:active_power", "sgen.7:connected", "sgen.7:is_connected", "sgen.7:active_power", "line.10:reactive_power_to", "line.10:current_to", "line.11:reactive_power_from", "line.11:current_from", "bus.14:voltage", "line.11:reactive_power_to", "line.11:current_to", "line.14:active_power_from", "line.14:reactive_power_from", "line.14:current_from", "line.14:voltage_from", "switch.31:closed", "switch.31:is_closed", "line.0:reactive_power_to", "line.1:reactive_power_from", "line.1:reactive_power_to", "line.2:reactive_power_from", "line.9:active_power_from", "line.9:reactive_power_from", "line.9:current_from", "line.2:reactive_power_to", "line.3:reactive_power_from", "sgen.1:target_active_power", "sgen.1:active_power", "line.3:reactive_power_to", "line.4:reactive_power_from", "line.4:voltage_from", "switch.16:closed", "switch.16:is_closed", "line.4:reactive_power_to", "line.4:voltage_to", "line.12:reactive_power_from", "switch.17:closed", "switch.17:is_closed", "line.5:active_power_from", "line.5:reactive_power_from", "line.5:current_from", "line.12:reactive_power_to", "line.5:active_power_to", "line.5:reactive_power_to", "line.5:current_to", "line.9:active_power_to", "line.9:reactive_power_to", "line.14:active_power_to", "line.14:reactive_power_to", "line.14:current_to", "line.14:voltage_to", "switch.4:closed", "switch.4:is_closed", "line.0:reactive_power_from", "trafo.0:reactive_power_lv", "line.6:reactive_power_to", "line.7:reactive_power_from", "line.7:reactive_power_to", "line.8:active_power_from", "line.8:reactive_power_from", "switch.23:closed", "switch.23:is_closed", "line.13:active_power_to", "line.13:reactive_power_to", "line.13:current_to", "line.6:reactive_power_from", "line.9:current_to", "switch.27:closed", "switch.27:is_closed"],

    "Gradient": [ "line.10:reactive_power_from", "trafo.1:reactive_power_lv", "line.10:reactive_power_to", "line.11:active_power_from", "line.11:reactive_power_from", "line.11:active_power_to", "line.11:reactive_power_to", "line.14:active_power_from", "line.14:reactive_power_from", "line.14:voltage_from", "line.14:active_power_to", "line.14:reactive_power_to", "line.14:voltage_to", "line.10:active_power_to", "line.10:active_power_from" ],

    "Steadytime": [ "trafo.0:tap_position", "sgen.10:target_active_power", "switch.23:closed", "switch.23:is_closed", "switch.25:closed", "switch.25:is_closed", "sgen.7:connected", "sgen.7:is_connected", "switch.31:closed", "switch.31:is_closed", "switch.2:closed", "switch.2:is_closed", "sgen.1:target_active_power", "switch.16:closed", "switch.16:is_closed", "switch.17:closed", "switch.17:is_closed", "switch.1:closed", "switch.1:is_closed", "switch.4:closed", "switch.4:is_closed", "switch.27:closed", "switch.27:is_closed" ],

    "Histogram": [ "sgen.10:target_active_power", "switch.23:closed", "switch.23:is_closed", "sgen.7:connected", "sgen.7:is_connected", "switch.31:closed", "switch.31:is_closed", "sgen.1:target_active_power", "switch.16:closed", "switch.16:is_closed", "switch.17:closed", "switch.17:is_closed", "switch.4:closed", "switch.4:is_closed", "switch.27:closed", "switch.27:is_closed" ],

    "PASAD": [],
}
IGNORE["All"] = list(set([i for ign in IGNORE.values() for i in ign]))

MINMAX = {
    "MinMax": {
        "_type": "MinMax",
        "model-file": "../models/MinMax",
        "features": [ ],
        "preprocessors": [],
        "trainon": 1.0,
        "save-training": None,
        "allow-none": False,
        "threshold": 1.0,
        "discrete_threshold": 10
    }
}

GRADIENT = {
    "Gradient": {
        "_type": "MinMax",
        "model-file": "../models/Gradient",
        "features": [ ],
        "preprocessors": [
            {"method": "Gradient", "features": []}
        ],
        "trainon": 1.0,
        "save-training": None,
        "allow-none": False,
        "threshold": 1.0,
        "discrete_threshold": 10
    }
}

STEAYDTIME = {
    "Steadytime": {
        "_type": "SteadyTime",
        "model-file": "../models/Steadytime",
        "features": [ ],
        "preprocessors": [],
        "trainon": 1.0,
        "save-training": None,
        "allow-none": False,
        "threshold": 1.0,
        "discrete_threshold": 10,
        "adjust": True,
    }
}

HISTOGRAM = {
    "Histogram": {
        "_type": "Histogram",
        "model-file": "../models/Histogram",
        "features": [ ],
        "preprocessors": [],
        "trainon": 1.0,
        "save-training": None,
        "allow-none": False,
        "window_size": 100,
        "threshold": 1.0,
        "discrete_threshold": 10,
    }
}

INVARIANT = {
    "Invariant": {
        "_type": "InvariantRules",
        "model-file": "../models/Invariant",
        "keyArray": [],
        "actuators": [],
        "discrete_threshold": 10,
        "eps": 0.01,
        "sigma": 1.1,
        "theta_value": 0.08,
        "gamma_value": 0.9,
        "max_k": 2,
        "max_comp": 3,
        "negated_states": 0,
        "merge_rules": 1,
        "parallel_filter": False,
        "mode": 2,
    }
}

SEQ2SEQNN = {
   "Seq2SeqNN": {
        "_type": "Seq2SeqNN",
        "model-file": "../models/Seq2SeqNN.gz",
        "features": [ ], # state;
        "preprocessors": [
            { "method": "MinMax", "features": [ ] }
        ],
        "trainon": 1.0,
        "save-training": None,
        "allow-none": False,
        "window_size": 100,
        "window_given": 90,
        "window_predict": 9,
        "batch_size": 4096,
        "gpu": "cpu",
        "threads": 4,
        "epochs": 150,
        "hidden_cells": 64,
        "n_hidden_cells": 64,
        "evaluation_norm_p": 4,
        "aggregate": 300,
        "cutoff": 60,
        "highpercentile": 90,
        "lowpercentile": 20,
        "alertratio": 20.0,
        "treshold": 0.3,
        "nn-model": None,
    }
}

GECO = {
    "GeCo" : {
        "_type": "GeCo",
        "model-file": "../models/GeCo",
        "ignore": [],
        "max_formel_length": 0,
        "threshold_factor": 1.0,
        "cusum_factor": 1.0,
        "cpus": 8,
    }
}

PASAD = {
    "PASAD": {
        "_type": "Pasad",
        "model-file": "./model",
        "N": 2000,
        "L": 250,
        "r": 45,
        "sensor": None,
        "start": 0,
        "apply_weights": False,
        "epsilon": None,
        "theta": None
    }
}

def open_file(filename, mode):
    if filename.endswith(".gz"):
        return gzip.open(filename, mode=mode)
    else:
        return open(filename, mode=mode, buffering=1)

def gen_config(filename):
    with open_file(filename, "r") as f:
        all_states = json.loads(f.readline())["state"].keys()

    # SIMPLE
    simple = {}

    # MinMax
    config = {}
    states = [s for s in all_states if s not in IGNORE["MinMax"]]
    for s in states:
        config[f"MinMax-{s}"] = copy.deepcopy(MINMAX["MinMax"])
        config[f"MinMax-{s}"]["features"] = [f"state;{s}"]
        config[f"MinMax-{s}"]["model-file"] = f"../models/MinMax-{s}"
    with open("config/MinMax.json", "w") as f:
        json.dump(config, f, indent=4),

    for k, v in config.items():
        simple[k] = copy.deepcopy(v)

    # Gradient
    config = {}
    states = [s for s in all_states if s not in IGNORE["Gradient"]]
    for s in states:
        config[f"Gradient-{s}"] = copy.deepcopy(GRADIENT["Gradient"])
        config[f"Gradient-{s}"]["features"] = [f"state;{s}"]
        config[f"Gradient-{s}"]["model-file"] = f"../models/Gradient-{s}"
        config[f"Gradient-{s}"]["preprocessors"][0]["features"].append(f"state;{s}")
    with open("config/Gradient.json", "w") as f:
        json.dump(config, f, indent=4)

    for k, v in config.items():
        simple[k] = copy.deepcopy(v)

    # Steadytime
    config = {}
    states = [s for s in all_states if s not in IGNORE["Steadytime"]]
    for s in states:
        config[f"Steadytime-{s}"] = copy.deepcopy(STEAYDTIME["Steadytime"])
        config[f"Steadytime-{s}"]["features"] = [f"state;{s}"]
        config[f"Steadytime-{s}"]["model-file"] = f"../models/Steadytime-{s}"
    with open("config/Steadytime.json", "w") as f:
        json.dump(config, f, indent=4)

    for k, v in config.items():
        simple[k] = copy.deepcopy(v)

    # Histogram
    config = {}
    states = [s for s in all_states if s not in IGNORE["Histogram"]]
    for s in states:
        config[f"Histogram-{s}"] = copy.deepcopy(HISTOGRAM["Histogram"])
        config[f"Histogram-{s}"]["features"] = [f"state;{s}"]
        config[f"Histogram-{s}"]["model-file"] = f"../models/Histogram-{s}"
    with open("config/Histogram.json", "w") as f:
        json.dump(config, f, indent=4)

    for k, v in config.items():
        simple[k] = copy.deepcopy(v)

    # SIMPLE
    with open("config/SIMPLE.json", "w") as f:
        json.dump(simple, f, indent=4)

    # Invariant
    config = copy.deepcopy(INVARIANT)
    states = [s for s in all_states if s not in IGNORE["All"]]
    config["Invariant"]["keyArray"].append([f"{s}" for s in states])
    with open("config/Invariant.json", "w") as f:
        json.dump(config, f, indent=4)

    # Seq2SeqNN
    config = copy.deepcopy(SEQ2SEQNN)
    states = [s for s in all_states if s not in IGNORE["All"]]
    config["Seq2SeqNN"]["features"] = [f"state;{s}" for s in states]
    config["Seq2SeqNN"]["preprocessors"][0]["features"] = [f"state;{s}" for s in states]
    with open("config/Seq2SeqNN.json", "w") as f:
        json.dump(config, f, indent=4)

    # Geco
    config = copy.deepcopy(GECO)
    config["GeCo"]["ignore"] = [ "bus.1:reactive_power", "line.10:active_power_from", "line.10:reactive_power_from", "line.10:current_from", "trafo.0:voltage_hv", "trafo.1:reactive_power_lv", "trafo.1:voltage_hv", "trafo.1:voltage_lv", "bus.9:reactive_power", "sgen.10:active_power", "sgen.11:active_power", "sgen.6:active_power", "bus.11:reactive_power", "sgen.7:active_power", "bus.13:voltage", "line.10:active_power_to", "line.10:reactive_power_to", "line.11:active_power_from", "line.11:reactive_power_from", "bus.14:voltage", "line.11:active_power_to", "line.11:reactive_power_to", "line.11:current_to", "line.14:active_power_from", "line.14:reactive_power_from", "line.14:current_from", "line.14:voltage_from", "bus.3:reactive_power", "line.1:reactive_power_to", "sgen.1:active_power", "storage.0:active_power", "line.5:active_power_from", "line.5:current_from", "line.5:active_power_to", "line.5:current_to", "line.14:active_power_to", "line.14:reactive_power_to", "line.14:current_to", "line.14:voltage_toline.14:active_power_to", "line.14:reactive_power_to", "line.14:current_to", "line.14:voltage_to" ]
    with open("config/GeCo.json", "w") as f:
        json.dump(config, f, indent=4)

    # PASAD
    config = {}
    states = [s for s in all_states if s not in IGNORE["PASAD"]]
    for s in states:
        config[f"PASAD-{s}"] = copy.deepcopy(PASAD["PASAD"])
        config[f"PASAD-{s}"]["sensor"] = f"{s}"
        config[f"PASAD-{s}"]["model-file"] = f"../models/PASAD-{s}"
    with open("config/PASAD.json", "w") as f:
        json.dump(config, f, indent=4),

    for k, v in config.items():
        simple[k] = copy.deepcopy(v)

def main():
    gen_config("./dataset/train.state.gz")

if __name__ == "__main__":
    main()
